#!/usr/bin/env python3
# -*- coding: utf-8 -*-
#************************************************************************
#
# Satellite: Sentinel-3A/B                                     Sensor: OLCI
#
# https://directory.eoportal.org/web/eoportal/satellite-missions/...
# https://www.wmo-sat.info/oscar/satellites/view/sentinel_3a/3b
#
# Orbit: Sun-synchronous app. LTDN=10:00   (Local Time on Descending Node)
#
#      **** This script takes one CLI parameter YYYYmmddNoD ****
#                       Where NoD must be DAY
#
# It only takes daylight passes       (OLCI does not gather any NIG data)
#------------------------------------------------------------------------
#
# Unfortunately OLCI EFR segments are not available on EUMETCast anymore.
# But we still get OLCI ERR passes. While this is no real replacement it
# is better than nothing. After all it's covering the whole planet like
# Metop GDS. There is no point in Sentinel-3A/B multipass as OLCI has very
# limited (west looking) swath. For more extended areas use Sentinel-3X
# that combines Sentinel-3A with Sentinel-3B passes in stacked images.
#
# CH-3123 Belp, 2022/12/10       License GPL3          (c) Ernst Lobsiger
#
#************************************************************************
#
# File Pattern : {mission_id:3s}_OL_1_{datatype_id:_<6s}_{start_time:%Y%m%dT%H%M%S}_{end_time:%Y%m%dT%H%M%S}_{creation_time:%Y%m%dT%H%M%S}_{duration:4d}_{cycle:3d}_{relative_orbit:3d}_{frame:4d}_{centre:3s}_{platform_mode:1s}_{timeliness:2s}_{collection:3s}.SEN3.tar
# Example name : S3A_OL_1_ERR____20220520T003318_20220520T011743_20220520T031351_2665_085_259______MAR_O_NR_002.SEN3.tar
# Hugo's ruler : 0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012
# copied EMCV  : 0         1         2         3         4         5         6         7         8         9         10
# EUMETCast    : These files arrive as above in channel E2H-S3A-02 (HVS-2)
# ANNOTATION   : This script uses ImageMagick (IM) convert for annotation
#************************************************************************
#
# File Pattern : {mission_id:3s}_OL_1_{datatype_id:_<6s}_{start_time:%Y%m%dT%H%M%S}_{end_time:%Y%m%dT%H%M%S}_{creation_time:%Y%m%dT%H%M%S}_{duration:4d}_{cycle:3d}_{relative_orbit:3d}_{frame:4d}_{centre:3s}_{platform_mode:1s}_{timeliness:2s}_{collection:3s}.SEN3.tar
# Example name : S3B_OL_1_ERR____20220520T003318_20220520T011743_20220520T031351_2665_085_259______MAR_O_NR_002.SEN3.tar
# Hugo's ruler : 0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012
# copied EMCV  : 0         1         2         3         4         5         6         7         8         9         10
# EUMETCast    : These files arrive as above in channel E2H-S3B-02 (HVS-2)
# ANNOTATION   : This script uses ImageMagick (IM) convert for annotation


# I need
from LEOstuff import test_argv, leo_images, get_magick
from LEO_extrastuff import get_args_leo,split_Dat,recreate_cmd,test_file_exist,satpy_viewer
import os, sys, platform

# Why to hell is it not working?
# from satpy.utils import debug_on
# debug_on()

# What you should know
OS = platform.system()

opts = get_args_leo('Sen3','A|B|X')

Yea, Mon, Day, Hou, Min=split_Dat(opts.Date)

sat='Sen3'+str(opts.sat)
if sat=='Sen3A':
  chan='E2H-S3A-02'
elif sat=='Sen3B':
  chan='E2H-S3B-02'
else:
  chan='unk'

curdir=os.getcwd()

# vvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvv
# ********** ADAPT THESE PARAMETERS TO YOUR PERSONAL NEEDS ***********

# Edit 0-4 parameter(s) below according to your file layout:
# My files are in a /MountPoint/Channel/YYYY/mm/dd structure
# Your segdir may be named differently from EUMETSAT channel
# No trailing / because this will make timestamps unreadable

if opts.src != '':
  segdir = opts.src
  if segdir[-1]=='/':
    segdir=segdir[:-1]
else:
  if OS == 'Linux':
      segdir = '/srv/'+chan
  elif OS == 'Windows':
      segdir = 'Z:/'+chan
  else:
      sys.exit('Sorry, OS ' + OS + ' seems unsupported yet ...')

if opts.bulk==None:      # use default value
  isbulk = True
else:
  isbulk = opts.bulk

decomp = False

if sat[-1:]=='X':
  segdir=[segdir,segdir]
  decomp=[decomp,decomp]
  isbulk=[isbulk,isbulk]

# Your TC receiver type
receiver = 'SR1'

# ********************************************************************************
# Avaliable composites: See files ../satpy/etc/composites/olci.yaml and visir.yaml
# ********************************************************************************
# composites = ['ocean_color', 'true_color', 'true_color_desert', 'true_color_land',
#              'true_color_marine_clean', 'true_color_marine_tropical', 'true_color_raw',
#              # These moisture composites are taken in from visir.yaml
#              'day_essl_colorized_low_level_moisture', 'day_essl_low_level_moisture',
#              'essl_colorized_low_level_moisture', 'essl_low_level_moisture',
#              # These composites have been used by several power users
#              # See the thread https://groups.io/g/MSG-1/message/30626
#              'ernst100603n', 'ernst100603o', 'ernst100603t',
#              'simon080503n', 'simon080503o', 'simon080503t',
#              'david090604n', 'david090604o', 'david090604t',
#              'hugo211709n',  'hugo211709o',  'hugo211709t',
#              'satpy080603n', 'satpy080603o', 'satpy080603t']


# Choose your composites for DAY (only) passes. These must be lists containing at least one entry.
if opts.composites=='':
  if opts.NoD == 'DAY':
      composites = ['ocean_color', 'true_color']
  else:   # 'NIG'
      sys.exit('Sorry, OLCI is a daylight instrument only ...')
else:
  composites=opts.composites.split(',')


kwargs = {'histtle':opts.histtle}
if opts.area=='':
  # Choose your 'area' name from file areas.yaml, there is no fake area 'swath' as in Starter Kit 3.0
  # Due to the shape of OLCI ERR 45 minute segment files, 'omerc_bb' 'laea_bb' areas cannot be chosen
  area = 'westminster'
else:
  area=opts.area

print('Area     : '+str(area))
print('Composite: '+str(composites))

multi = opts.mp
#multi=False   # no multipass with sentinel? Gives error message!
#In LEOstuff.py:
#  Satnumber is not used here as one segment is equal to one pass
#  ==> Satnumber[n] gaat mis
#
# Optionally you can define an area dependant cities-list. If empty a sat default cities-list is used.
# Be aware that adding one city name will change your OVRcache even if the city is outside your image.
area_cities = []

# Configure your individual overlays, either True/False or 1/0 do work
if opts.default:               # default overlays, no -ovl option
  ADDcoasts  = True
  ADDborders = True
  ADDrivers  = True
  ADDlakes   = True
  ADDgrid    = True
  ADDcities  = True
  ADDpoints  = True
  ADDstation = True
  ADDlegend  = True
else:                          # take overlays from option -ovl
  ADDcoasts  = opts.coasts
  ADDborders = opts.borders
  ADDrivers  = opts.rivers
  ADDlakes   = opts.lakes
  ADDgrid    = opts.grid
  ADDcities  = opts.cities
  ADDpoints  = opts.points
  ADDstation = opts.station
  ADDlegend  = opts.legend

# OVRCache speeds up static overlays. These are stored in your .../SPSdata/cache
# Never use this with homebrew or SatPy's dynamic areas like omec_bb and laea_bb
# With caching you will sometimes have to delete *.png OVRcache files manually !
OVRcache = True

# If you are testing interactively while PyTROLL/SatPy image generation croned
# This allows for one interactively started script that will use tmpdirs/xtest
testrun = False

subdir = opts.NoD

#Only if filename is given: test if file exists, don't overwrite
if opts.ofile!='':
  pfn=test_file_exist(sat,opts.dst,subdir,opts.ofile)
  if pfn != '':
    print(pfn+' exists, nothing done.')
    quit()

# ******** TOUCH THE CODE BELOW ONLY IF YOU KNOW WHAT YOU DO *********
# ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^


Yea, Mon, Day, Hou, Min, height = leo_images(Yea, Mon, Day, sat, opts.NoD, multi, segdir, decomp, isbulk, 'olci_l1b',
                                             composites, area, area_cities, ADDcoasts, ADDborders, ADDrivers, ADDlakes,
                                             ADDgrid, ADDcities, ADDpoints, ADDstation, OVRcache, testrun, **kwargs)


# *******************************************************************
# POST PROCESSING LEGEND: WORKS WITH ALL SCRIPTS AND PROJECTION AREAS
# *******************************************************************

# Below is for legend only
sensor = 'olci'
# EUMETCast Europe service
service = 'HVS-2'
# EUMETSAT channel naming
channel = chan
# Logos in logos directory
logo1 = 'Copernicus_1050x1050.png'
logo2 = 'PyTROLL_400x400.jpg'

for composite in composites:

    magick = get_magick(Yea, Mon, Day, Hou, Min, height, logo1, logo2, service, channel, receiver,
                        sat, sensor, composite, area, testrun, opts.NoD, False, ADDlegend,
                        subdir=subdir)  # basedir=basedir, subdir=subdir

    magick,pfn=recreate_cmd(magick,opts.dst,opts.ofile,curdir)
    # **ImageMagick**
    os.system(magick)
    print('Generated '+pfn)
    if opts.do_view:
      satpy_viewer(pfn)

# If you get an image in the satellites tmp directory but no final image with
# legend at left then you have a problem with ImageMagick. THAT'S ALL FOLKS?!

