#!/usr/bin/env python3
# -*- coding: utf-8 -*-
#************************************************************************
#
# Satellite: EOS-Aqua                                       Sensor: MODIS
# https://directory.eoportal.org/web/eoportal/satellite-missions/a/aqua
# https://www.wmo-sat.info/oscar/satellites/view/aqua
#
# Orbit: Sun-synchronous app. LTAN=13:30   (Local Time on Ascending Node)
#
#          ***** This script takes an argument YYYYmmddNoD *****
#                   Where NoD must be either NIG or DAY
#
# Satellite: EOS-Terra                                      Sensor: MODIS
# https://directory.eoportal.org/web/eoportal/satellite-missions/t/terra
# https://www.wmo-sat.info/oscar/satellites/view/terra
#
# Orbit: Sun-synchronous app. LTDN=10:30  (Local Time on Descending Node)
#
#         ***** This script takes an argument YYYYmmddNoD *****
#                  Where NoD must be either NIG or DAY
#------------------------------------------------------------------------
#
# CH-3123 Belp, 2022/12/10       License GPL3          (c) Ernst Lobsiger
#
#************************************************************************
#
# File Pattern : thin_M{platform_indicator:1s}D021KM.A{start_time:%Y%j.%H%M}.{collection:03d}{suffix}.hdf
# Example name : thin_MYD021KM.A2022203.0450.061.2022203061813.NRT.hdf
# Hugo's ruler : 01234567890123456789012345678901234567890123456789012
# copied EMCV  : 0         1         2         3         4         5
# EUMETCast    : These files arrive as above in channel E1B-TPL-1 (Basic Service)
# ANNOTATION   : This script uses ImageMagick (IM) convert for annotation
#
# File Pattern : thin_M{platform_indicator:1s}D021KM.A{start_time:%Y%j.%H%M}.{collection:03d}{suffix}.hdf
# Example name : thin_MOD021KM.A2022203.0050.061.2022203015550.NRT.hdf
# Hugo's ruler : 01234567890123456789012345678901234567890123456789012
# copied EMCV  : 0         1         2         3         4         5
# EUMETCast    : These files arrive as above in channel E1B-TPL-1 (Basic Service)
# ANNOTATION   : This script uses ImageMagick (IM) convert for annotation

# I need
from LEOstuff import test_argv, leo_images, get_magick
from LEO_extrastuff import get_args_leo,split_Dat,recreate_cmd,test_file_exist,satpy_viewer
import os, sys, platform

# Why to hell is it not working?
# from satpy.utils import debug_on
# debug_on()

# What you should know
OS = platform.system()

opts = get_args_leo('AquaTerra','A|T')

Yea, Mon, Day, Hou, Min=split_Dat(opts.Date)

if opts.sat=='A':
  sat='Aqua'
else:
  sat='Terra'

curdir=os.getcwd()

# vvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvv
# ********** ADAPT THESE PARAMETERS TO YOUR PERSONAL NEEDS ***********

# Edit 0-4 parameter(s) below according to your file layout:
# My files are in a /MountPoint/Channel/YYYY/mm/dd structure
# Your segdir may be named differently from EUMETSAT channel
# No trailing / because this will make timestamps unreadable
chan='E1B-TPL-1'
if opts.src != '':
  segdir = opts.src
  if segdir[-1]=='/':
    segdir=segdir[:-1]
else:
  if OS == 'Linux':
      segdir = '/srv/'+chan
  elif OS == 'Windows':
      segdir = 'Z:/'+chan
  else:
      sys.exit('Sorry, OS ' + OS + ' seems unsupported yet ...')

if opts.bulk==None:      # use default value
  isbulk = True
else:
  isbulk = opts.bulk

decomp = False

# Your TC receiver type
receiver = 'SR1'

# Available composites in this script   (see .../satpy/etc/composites/modis/visir.yaml):
# **************************************************************************************
# ['ash', 'day_microphysics', 'dust', 'fog', 'green_snow', 'ir108_3d', 'ir_cloud_day', 'natural_color',
#  'natural_color_raw', 'natural_with_night_fog', 'night_fog', 'overview', 'snow', 'true_color_thin']

# Choose your composites for DAY and NIG passes. These must be lists containing at least one entry.
if opts.composites=='':
  if opts.NoD == 'DAY':
      composites = ['natural_color', 'overview', 'true_color_thin']
  else:   # 'NIG'
      composites = ['dust', 'ir108_3d', 'night_fog']
else:
  composites=opts.composites.split(',')

kwargs = {'histtle':opts.histtle}
if opts.area=='':
  # Choose your 'area' name from file areas.yaml, there is no fake area 'swath' as in Starter Kit 3.0
  # For dynamic areas 'omerc_bb', 'laea_bb' set your POI with kwargs, else POI defaults to your station
  # kwargs = {'lon': 7.5, 'lat': 47.0, 'ran': 15.0}   <--- This is an example, kwargs is a dictionary
  area = 'westminster'
else:
  area=opts.area

print('Area     : '+str(area))
print('Composite: '+str(composites))

# Depending on the size of your area you may want to display multiple passes (e.g. for area = 'eurol')
# You cannot choose multi = True for areas 'omerc_bb' and 'laea_bb' that work for single passes only!!
multi = opts.mp

# Optionally you can define an area dependant cities-list. If empty a sat default cities-list is used.
# Be aware that adding one city name will change your OVRcache even if the city is outside your image.
area_cities = []

# Configure your individual overlays, either True/False or 1/0 do work
if opts.default:               # default overlays, no -ovl option
  ADDcoasts  = True
  ADDborders = True
  ADDrivers  = True
  ADDlakes   = True
  ADDgrid    = True
  ADDcities  = True
  ADDpoints  = True
  ADDstation = True
  ADDlegend  = True
else:                          # take overlays from option -ovl
  ADDcoasts  = opts.coasts
  ADDborders = opts.borders
  ADDrivers  = opts.rivers
  ADDlakes   = opts.lakes
  ADDgrid    = opts.grid
  ADDcities  = opts.cities
  ADDpoints  = opts.points
  ADDstation = opts.station
  ADDlegend  = opts.legend

# OVRCache speeds up static overlays. These are stored in your .../SPSdata/cache
# Never use this with homebrew or SatPy's dynamic areas like omec_bb and laea_bb
# With caching you will sometimes have to delete *.png OVRcache files manually !
OVRcache = True

# If you are testing interactively while PyTROLL/SatPy image generation croned
# This allows for one interactively started script that will use tmpdirs/xtest
testrun = False

subdir = opts.NoD

#Only if filename is given: test if file exists, don't overwrite
if opts.ofile!='':
  pfn=test_file_exist(sat,opts.dst,subdir,opts.ofile)
  print(pfn)
  if pfn != '':
    print(pfn+' exists, nothing done.')
    quit()

# ******** TOUCH THE CODE BELOW ONLY IF YOU KNOW WHAT YOU DO *********
# ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^

Yea, Mon, Day, Hou, Min, height = leo_images(Yea, Mon, Day, sat, opts.NoD, multi, segdir, True, isbulk, 'modis_l1b',
                                             composites, area, area_cities, ADDcoasts, ADDborders, ADDrivers, ADDlakes,
                                             ADDgrid, ADDcities, ADDpoints, ADDstation, OVRcache, testrun, **kwargs)


# *******************************************************************
# POST PROCESSING LEGEND: WORKS WITH ALL SCRIPTS AND PROJECTION AREAS
# *******************************************************************

# Below is for legend only
sensor = 'modis'
# EUMETCast Europe service
service = 'Basic'
# EUMETSAT channel naming
channel = chan
# Logos in logos directory
logo1 = 'NOAA_1200x1200.png'
logo2 = 'PyTROLL_400x400.jpg'

for composite in composites:

    magick = get_magick(Yea, Mon, Day, Hou, Min, height, logo1, logo2, service, channel, receiver,
                        sat, sensor, composite, area, testrun, opts.NoD, multi, ADDlegend,
                        subdir = subdir)  # basedir=basedir, subdir=subdir

    magick,pfn=recreate_cmd(magick,opts.dst,opts.ofile,curdir)
    # **ImageMagick**
    os.system(magick)
    print('Generated '+pfn)
    if opts.do_view:
      satpy_viewer(pfn)

# If you get an image in the satellites tmp directory but no final image with
# legend at left then you have a problem with ImageMagick. THAT'S ALL FOLKS?!

